function setDepParams(bv)
%SETBVPARAMS Set the dependent parameters for the Vanthoor (2011) model
% Dependent parameters are parameters that depend on the setting of another
% parameter
% Inputs:
%   bv   - a StateSpaceModel object to be used as a Vanthoor model
%
% Based on the electronic appendices (the case of a Dutch greenhouse) of:
%   [1] Vanthoor, B., Stanghellini, C., van Henten, E. J. & de Visser, P. H. B. 
%       A methodology for model-based greenhouse design: Part 1, a greenhouse climate 
%       model for a broad range of designs and climates. Biosyst. Eng. 110, 363377 (2011).
%   [2] Vanthoor, B., de Visser, P. H. B., Stanghellini, C. & van Henten, E. J. 
%       A methodology for model-based greenhouse design: Part 2, description and 
%       validation of a tomato yield model. Biosyst. Eng. 110, 378-395 (2011).

% David Katzin, Wageningen University
% david.katzin@wur.nl

	% Heat capacity of heating pipes [J K^{-1} m^{-2}]
    % Equation 21 [1]
    bv.p.capPipe = 0.25*pi*bv.p.lPipe*((bv.p.phiPipeE^2-bv.p.phiPipeI^2)*bv.p.rhoSteel*...
        bv.p.cPSteel+bv.p.phiPipeI^2*bv.p.rhoWater*bv.p.cPWater);
    
	% Density of the air [kg m^{-3}]
	% Equation 23 [1]
    bv.p.rhoAir = bv.p.rhoAir0*exp(bv.p.g*bv.p.mAir*bv.p.hElevation/(293.15*bv.p.R));
    
	% Heat capacity of greenhouse objects [J K^{-1} m^{-2}]
	% Equation 22 [1]
    bv.p.capAir = bv.p.hAir*bv.p.rhoAir*bv.p.cPAir;             % air in main compartment
    bv.p.capFlr = bv.p.hFlr*bv.p.rhoFlr*bv.p.cPFlr;             % floor
    bv.p.capSo1 = bv.p.hSo1*bv.p.rhoCpSo;                       % soil layer 1
    bv.p.capSo2 = bv.p.hSo2*bv.p.rhoCpSo;                       % soil layer 2
    bv.p.capSo3 = bv.p.hSo3*bv.p.rhoCpSo;                       % soil layer 3
    bv.p.capSo4 = bv.p.hSo4*bv.p.rhoCpSo;                       % soil layer 4
    bv.p.capSo5 = bv.p.hSo5*bv.p.rhoCpSo;                       % soil layer 5
    bv.p.capThScr = bv.p.hThScr*bv.p.rhoThScr*bv.p.cPThScr;     % thermal screen
    bv.p.capTop = (bv.p.hGh-bv.p.hAir)*bv.p.rhoAir*bv.p.cPAir;  % air in top compartments
    
    bv.p.capBlScr = bv.p.hBlScr*bv.p.rhoBlScr*bv.p.cPBlScr;     % blackout screen

    
	% Capacity for CO2 [m]
	bv.p.capCo2Air = bv.p.hAir;
    bv.p.capCo2Top = bv.p.hGh-bv.p.hAir;
	
	% Surface of pipes for floor area [-]
	% Table 3 [1]
    bv.p.aPipe = pi*bv.p.lPipe*bv.p.phiPipeE;
	
	% View factor from canopy to floor
	% Table 3 [1]
    bv.p.fCanFlr = 1-0.49*pi*bv.p.lPipe*bv.p.phiPipeE;
    
    % Absolute air pressure at given elevation [Pa]
	% See https://www.engineeringtoolbox.com/air-altitude-pressure-d_462.html
    bv.p.pressure = 101325*(1-2.5577e-5*bv.p.hElevation)^5.25588;

    bv.p.cLeafMax = bv.p.laiMax./bv.p.sla;      % maximum leaf size                                                                             mg{leaf} m^{-2}

    bv.p.rFruit = bv.p.cFruitG*bv.p.fFruit;     % Growth respiration coefficienct of fruit, assuming constant fraction of fruit                 -
    bv.p.rStem = bv.p.cStemG*bv.p.fStem;        % Growth respiration coefficienct of stem, assuming constant fraction of stem                   -
    bv.p.rLeaf = bv.p.cLeafG*bv.p.fLeaf;        % Growth respiration coefficienct of leaves, assuming constant fraction of leaves               -
    
    bv.p.aGroPipe = pi*bv.p.lGroPipe*bv.p.phiGroPipeE; % Surface area of pipes for floor area                                                   m^{2}{pipe} m^{-2}{floor}
    
    % Heat capacity of grow pipes [J K^{-1} m^{-2}]
    % Equation 21 [1]
    bv.p.capGroPipe = 0.25*pi*bv.p.lGroPipe*((bv.p.phiGroPipeE^2-bv.p.phiGroPipeI^2)*bv.p.rhoSteel*...
        bv.p.cPSteel+bv.p.phiGroPipeI^2*bv.p.rhoWater*bv.p.cPWater);
end

